/***************************************************************************
 *   Copyright (C) 2011 by Jeremy Burton   *
 *   jburton@39net-w04   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <math.h>
#include <QPainter>

#include <stdio.h>

#include "dom.h"
#include "boxitem.h"

boxitem::boxitem() : baseitem()
{
  width=height=150.0;
  strokewidth=1.0;
  square=true;
  constrainprop=true;
  fill=false;
  fillcolor=Qt::white;
}

////////////////////////////////////////////////////////////////////////////////

baseitem *boxitem::dup()
{
  boxitem *item = new boxitem;
  *item = *this;
  return item;
}

////////////////////////////////////////////////////////////////////////////////

double boxitem::extent()
{
	CPoint p((width+strokewidth)/2.0,(width+strokewidth)/2.0);
	p.rotate(angle);
	return std::max(abs(p.x()),abs(p.y()));
}

////////////////////////////////////////////////////////////////////////////////

QString boxitem::describe()
{
  if (square)
    return QString("Box, square, %1mm").arg(width,0,'f',1);
  else
    return QString("Box, %1 x %2mm").arg(width,0,'f',1).arg(height,0,'f',1);
}

////////////////////////////////////////////////////////////////////////////////

void boxitem::toDom(QDomDocument &doc,QDomElement &parent)
{
  QDomElement aw = doc.createElement( "boxitem" );

  aw.setAttribute(QString("square"),square);
  aw.setAttribute(QString("constrainprop"),constrainprop);

  baseitem::toDom_content(doc,aw);

  toDom_Text(doc,aw,"width",width);
  toDom_Text(doc,aw,"height",height);

  parent.appendChild(aw);
}

////////////////////////////////////////////////////////////////////////////////

void boxitem::fromDom(QDomDocument &doc,QDomElement &element)
{
  baseitem::fromDom(doc,element);

  square = element.attribute("square", "1").toInt() != 0;
  constrainprop = element.attribute("constrainprop", "1").toInt() != 0;

  QDomNode n = element.firstChild();
  while(!n.isNull()) 
  {
    QDomElement e = n.toElement(); // try to convert the node to an element.
    if(!e.isNull())
    {
      if (e.tagName() == "width")
        width = e.text().toDouble();
      else if (e.tagName() == "height")
	height = e.text().toDouble();
    }
    n = n.nextSibling();
  }
}

////////////////////////////////////////////////////////////////////////////////

void boxitem::paint_handles(QPainter &painter, double zoom)
{
  QPoint
    ctr(centre.x()*zoom,-centre.y()*zoom),
    ctr1(0,0),
    rad(ctr1.x()+width/2.0*zoom,ctr1.y()-height/2.0*zoom);

  move_handle = ctr;
  angle_handle.setPolar(ctr,height/2.0*zoom+ANGLE_HANDLE_EXTEND,angle);
  size_handle = ctr+rad;
  size_handle.rotate(ctr,angle);

  painter.translate(ctr);
  painter.rotate(angle);
  paint_handle_move(painter,ctr1);
  paint_handle_size(painter,rad);
  paint_handle_angle2(painter,ctr1,height/2.0*zoom);
}

////////////////////////////////////////////////////////////////////////////////

void boxitem::paint(QPainter &painter, bool selected, double zoom)
{
  //int
    //dpix = painter.device()->logicalDpiX(),
    //dpiy = painter.device()->logicalDpiY();
  int
    w=width*zoom,
    h=height*zoom;
  QPoint
    ctr(centre.x()*zoom,-centre.y()*zoom);
  QPen
    pen1(Qt::SolidLine);
  QColor
    scolor(strokecolor),
    fcolor(fillcolor);

  if (!selected)
  {
    scolor.setAlpha(127);
    fcolor.setAlpha(127);
  }

  if (stroke)
  {
    pen1.setWidth(strokewidth*zoom);
    pen1.setJoinStyle(Qt::MiterJoin);
    pen1.setColor(scolor);
    painter.setPen(pen1);
  }
  else
    painter.setPen(Qt::NoPen);

  if (fill)
    painter.setBrush(fcolor);
  else
    painter.setBrush(Qt::NoBrush);

  painter.translate(ctr);
  painter.rotate(angle);
  painter.drawRect(-w/2.0,-h/2.0,w,h);
}

////////////////////////////////////////////////////////////////////////////////

void boxitem::output(QPainter &painter)
{
  int
    dpix = painter.device()->logicalDpiX(),
    dpiy = painter.device()->logicalDpiY(),
    w = realtodev(width,dpix),
    h = realtodev(height,dpiy);
  QPoint
    ctr(realtodev(centre.x(),dpix),realtodev(-centre.y(),dpiy));
  QPen
    pen1(strokecolor);

//printf("DPI=%d,%d\n",dpix,dpiy);

  if (stroke)
  {
    pen1.setWidth(realtodev(strokewidth,dpix));
    pen1.setJoinStyle(Qt::MiterJoin);
    painter.setPen(pen1);
  }
  else
    painter.setPen(Qt::NoPen);

  if (fill)
    painter.setBrush(fillcolor);
  else
    painter.setBrush(Qt::NoBrush);

  painter.translate(ctr);
  painter.rotate(angle);
  painter.drawRect(-w/2.0,-h/2.0,w,h);
}

////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

