/***************************************************************************
 *   Copyright (C) 2011 by Jeremy Burton   *
 *   jburton@39net-w04   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include <math.h>
#include <QPainter>

#include <stdio.h>

#include "dom.h"
#include "ellipseitem.h"

ellipseitem::ellipseitem() : polaritem()
{
  circular=false;
  radius2=25.0;
  strokewidth=1.0;
}

////////////////////////////////////////////////////////////////////////////////

baseitem *ellipseitem::dup()
{
  ellipseitem *item = new ellipseitem;
  *item = *this;
  return item;
}

////////////////////////////////////////////////////////////////////////////////

double ellipseitem::extent()
{
  return std::max(radius1,radius2)+strokewidth/2.0;
}

////////////////////////////////////////////////////////////////////////////////

QString ellipseitem::describe()
{
  if (circular)
    return QString("Circle, r %1mm").arg(radius1,0,'f',1);
  else
    return QString("Ellipse, r %1 x %2mm").arg(radius1,0,'f',1).arg(radius2,0,'f',1);
}

void ellipseitem::toDom(QDomDocument &doc,QDomElement &parent)
{
  QDomElement aw = doc.createElement( "ellipseitem" );

  polaritem::toDom_content(doc,aw);

  //toDom_Text(doc,aw,"radius2",radius2);

  parent.appendChild(aw);
}

void ellipseitem::fromDom(QDomDocument &doc,QDomElement &element)
{
  polaritem::fromDom(doc,element);

/*
  QDomNode n = element.firstChild();
  while(!n.isNull()) 
  {
    QDomElement e = n.toElement(); // try to convert the node to an element.
    if(!e.isNull())
    {
      if (e.tagName() == "radius2")
	radius2 = e.text().toDouble();
    }
    n = n.nextSibling();
  }
*/
}

void ellipseitem::paint_handles(QPainter &painter, double zoom)
{
  int
    rx1 = radius1*zoom,
    ry1 = radius2*zoom,
    rah = ry1+ANGLE_HANDLE_EXTEND;
  QPoint
    ctr(centre.x()*zoom,-centre.y()*zoom),
    ctr1(0,0);
  QPen
    pen1(Qt::DashLine);

  painter.translate(ctr);
  if (!circular)
    painter.rotate(angle);

  //Cross axes...
  pen1.setColor(Qt::gray);
  painter.setPen(pen1);
  painter.drawLine(0,-ry1,0,ry1);
  painter.drawLine(-rx1,0,rx1,0);

  //Ellipse Radii...
  pen1.setColor(Qt::blue);
  painter.setPen(pen1);
  painter.setBrush(Qt::NoBrush);
  painter.drawEllipse(ctr1,rx1,ry1);

  //Use NON-rotated,non-item centred canvas to set handle points
  if (!centrelock)
  {
    move_handle = ctr;
    paint_handle_move(painter,ctr1);
  }
  if (!anglelock && !circular)
  {
    angle_handle.setPolar(ctr,rah,angle);
    paint_handle_angle2(painter,ctr1,ry1);
  }
  if (!sizelock)
  {
    if (circular)
    {
      radius1_handle.setPolar(ctr,rx1,ry1,45.0);
      paint_handle_size(painter,ctr1,rx1,ry1,45.0);
    }
    else
    {
      radius1_handle.setPolar(ctr,rx1,ry1,90.0);
      radius1_handle.rotate(ctr,angle);
      radius2_handle.setPolar(ctr,rx1,ry1,0.0);
      radius2_handle.rotate(ctr,angle);
      paint_handle_size(painter,ctr1,rx1,ry1,90.0);
      paint_handle_size(painter,ctr1,rx1,ry1,0.0);
    }
  }
}

void ellipseitem::paint(QPainter &painter, bool selected, double zoom)
{
  int
    rx = radius1*zoom,
    ry = radius2*zoom;
  QPoint
    ctr(centre.x()*zoom,-centre.y()*zoom),
    ctr1(0,0);
  QPen
    pen1(Qt::SolidLine);
  QColor
    scolor(strokecolor),
    fcolor(fillcolor);

  if (!selected)
  {
    scolor.setAlpha(127);
    fcolor.setAlpha(127);
  }

  if (stroke)
  {
    pen1.setWidth(strokewidth*zoom);
    pen1.setJoinStyle(Qt::MiterJoin);
    pen1.setColor(scolor);
    painter.setPen(pen1);
  }
  else
    painter.setPen(Qt::NoPen);

  if (fill)
    painter.setBrush(fcolor);
  else
    painter.setBrush(Qt::NoBrush);

  painter.translate(ctr);
  painter.rotate(angle);
  painter.drawEllipse(ctr1,rx,ry);
}

void ellipseitem::output(QPainter &painter)
{
  int
    dpix = painter.device()->logicalDpiX(),
    dpiy = painter.device()->logicalDpiY(),
    rx = realtodev(radius1,dpix),
    ry = realtodev(radius2,dpiy);
  QPoint
    ctr(realtodev(centre.x(),dpix),realtodev(-centre.y(),dpiy)),
    ctr1(0,0);
  QPen
    pen1(strokecolor);

  if (stroke)
  {
    pen1.setWidth(realtodev(strokewidth,dpix));
    pen1.setJoinStyle(Qt::MiterJoin);
    painter.setPen(pen1);
  }
  else
    painter.setPen(Qt::NoPen);

  if (fill)
    painter.setBrush(fillcolor);
  else
    painter.setBrush(Qt::NoBrush);

  painter.translate(ctr);
  painter.rotate(angle);
  painter.drawEllipse(ctr1,rx,ry);
}

