#include <lxpanel/plugin.h>

#include <stdio.h>
#include <errno.h>
#include <sys/socket.h>
#include <string.h>
#include <net/if.h>
#include <netinet/in.h>
#include <stdlib.h>

#include <linux/netlink.h>
#include <linux/rtnetlink.h>

#define MYPROTO NETLINK_ROUTE
#define MYMGRP RTMGRP_LINK|RTMGRP_IPV4_IFADDR|RTMGRP_IPV6_IFADDR

/*
gcc -Wall `pkg-config --cflags gtk+-2.0 lxpanel` -shared -fPIC vpnmon.c -o vpnmon.so `pkg-config --libs lxpanel`
cp vpnmon.so /usr/lib64/lxpanel/plugins
*/

// internal to the plugin source, not used by the 'priv' variable
static int iInstanceCount = 0;

/* the plugin's id – an instance of this struct 
   is what will be assigned to 'priv' */
typedef struct 
{
    gint iMyId;
    //GtkWidget *mainw;
    GtkWidget *pLabel;
    guint ttag;
    
    int nls;
} VPNMon_data_s;

/**********************************************************************************************************************/

static int open_netlink()
{
    int sock = socket(AF_NETLINK,SOCK_RAW,MYPROTO);
    struct sockaddr_nl addr;

    memset((void *)&addr, 0, sizeof(addr));

    if (sock<0)
        return sock;
    addr.nl_family = AF_NETLINK;
    addr.nl_pid = getpid();
    addr.nl_groups = MYMGRP;
    if (bind(sock,(struct sockaddr *)&addr,sizeof(addr))<0)
        return -1;
    return sock;
}

static void close_netlink(int nls)
{
    if (nls>=0)
	close(nls);
}

static int read_events(int sockint, int (*msg_handler)(struct sockaddr_nl *,
                                               struct nlmsghdr *))
{
    int status;
    int ret = 0;
    char buf[4096];
    struct iovec iov = { buf, sizeof buf };
    struct sockaddr_nl snl;
    struct msghdr msg = { (void*)&snl, sizeof snl, &iov, 1, NULL, 0, 0};
    struct nlmsghdr *h;

    for (;;)
    {
    status = recvmsg(sockint, &msg, MSG_DONTWAIT);
//printf("read_netlink: recvmsg=%d\n", status);

    if (status < 0)
    {
//printf("read_netlink: errno=%d\n", errno);
        /* Socket non-blocking so bail out once we have read everything */
        if (errno == EWOULDBLOCK || errno == EAGAIN)
            return status;
	else
	{
	    /* Anything else is an error */
	    printf("read_netlink: Error recvmsg: %d\n", status);
	    perror("read_netlink: Error: ");
	    return status;
	}
    }

    if(status == 0)
    {
        printf("read_netlink: EOF\n");
	return status;
    }

    /* We need to handle more than one message per 'recvmsg' */
    for(h = (struct nlmsghdr *) buf; NLMSG_OK (h, (unsigned int)status); h = NLMSG_NEXT (h, status))
    {
        /* Finish reading */
        if (h->nlmsg_type == NLMSG_DONE)
            break;

        /* Message is some kind of error */
        if (h->nlmsg_type == NLMSG_ERROR)
        {
            printf("read_netlink: Message is an error - decode TBD\n");
            break; // Error
        }

        /* Call message handler */
        if(msg_handler)
        {
            ret = (*msg_handler)(&snl, h);
            if(ret < 0)
            {
                printf("read_netlink: Message hander error %d\n", ret);
            }
        }
        else
        {
            printf("read_netlink: Error NULL message handler\n");
        }
    }
    }
    return ret;
}

static struct if_data_s
{
    int index;
    char name[32];
    unsigned int up : 1;
} *if_data = NULL;
static int if_data_count = 0;

static struct if_data_s *get_if_data(int ifindex)
{
    int i;
    
    for (i=0; i<if_data_count; i++)
	if (if_data[i].index == ifindex)
	    return &if_data[i];
    return NULL;
}

static void set_if_data(int ifindex,int ifflags)
{
    struct if_data_s *p;
    char ifname[IF_NAMESIZE];
    
    p = get_if_data(ifindex);
    if (!p)
    {
	if_data_count++;
	if_data = (struct if_data_s *)realloc(if_data, sizeof(struct if_data_s)*if_data_count);
	p = &if_data[if_data_count-1];
	p->index = ifindex;
    }
    if (if_indextoname(ifindex,ifname))
	strncpy(p->name, ifname, 32);
    if (ifflags>=0)
	p->up = ifflags & IFF_UP;
}

static void del_if_data(int ifindex)
{
    struct if_data_s *p;
    
    p = get_if_data(ifindex);
    if (p)
    {
	memcpy(p,p+1,sizeof(struct if_data_s));
	if_data_count--;
	if_data = (struct if_data_s *)realloc(if_data, sizeof(struct if_data_s)*if_data_count);
    }
}

static int netlink_link_state(struct sockaddr_nl *nl, struct nlmsghdr *msg)
{
    //int len;
    struct ifinfomsg *ifi;

    nl = nl;

    ifi = NLMSG_DATA(msg);
    char ifname[1024];if_indextoname(ifi->ifi_index,ifname);

    printf("netlink_link_state: Link %s %s\n", 
            /*(ifi->ifi_flags & IFF_RUNNING)?"Up":"Down");*/
        ifname,(ifi->ifi_flags & IFF_UP)?"Up":"Down");
    return 0;
}

static int msg_handler(struct sockaddr_nl *nl, struct nlmsghdr *msg)
{
    struct ifinfomsg *ifi=NLMSG_DATA(msg);
    struct ifaddrmsg *ifa=NLMSG_DATA(msg);
    char ifname[1024]="\0";
    struct if_data_s *p;
    
    switch (msg->nlmsg_type)
    {
        case RTM_NEWADDR:
            if_indextoname(ifa->ifa_index,ifname);
            printf("msg_handler: RTM_NEWADDR : %s\n",ifname);
            break;
        case RTM_DELADDR:
            if_indextoname(ifa->ifa_index,ifname);
            printf("msg_handler: RTM_DELADDR : %s\n",ifname);
            break;
        case RTM_NEWLINK:
            if (if_indextoname(ifi->ifi_index,ifname))
		set_if_data(ifi->ifi_index,ifi->ifi_flags);
printf("msg_handler: ifi_index=%d\n", ifi->ifi_index);
            printf("msg_handler: RTM_NEWLINK\n");
            netlink_link_state(nl, msg);
            break;
        case RTM_DELLINK:
	    if (p = get_if_data(ifi->ifi_index))
		strcpy(ifname,p->name);
printf("msg_handler: ifi_index=%d\n", ifi->ifi_index);
            netlink_link_state(nl, msg);
            printf("msg_handler: RTM_DELLINK : %s\n",ifname);
	    del_if_data(ifi->ifi_index);
            break;
        default:
            printf("msg_handler: Unknown netlink nlmsg_type %d\n",
                    msg->nlmsg_type);
            break;
    }
    return 0;
}

/**********************************************************************************************************************/

static gboolean VPNMon_timer1(VPNMon_data_s *VPNMon_data)
{
    static int i=0;
    
    if (VPNMon_data->nls>=0)
      read_events(VPNMon_data->nls, msg_handler);

    char cIdBuf[4] = {'\0'};

    snprintf(cIdBuf, sizeof(cIdBuf), "%d", i);
    
    gtk_label_set_text((GtkLabel *)(VPNMon_data->pLabel),cIdBuf);
    
    i=(i+1) % 10;
 //printf("timer1: ttag=%d\n",(int)(VPNMon_data->ttag));
   
    return TRUE;
}

static void VPNMon_destructor(gpointer user_data)
{
    VPNMon_data_s *VPNMon_data = (VPNMon_data_s *)user_data;

    close_netlink(VPNMon_data->nls);
    
printf("test_destructor: \n\tttag=%d\n\tpLabel=%p\n",(int)(VPNMon_data->ttag),VPNMon_data->pLabel);
    g_source_remove(VPNMon_data->ttag);
    
//     netproc_netdevlist_clear(&ns->fnetd->netdevlist);
//     /* The widget is destroyed in plugin_stop().
//     gtk_widget_destroy(ns->mainw);
//     */
//     lxnm_close(ns->fnetd->lxnmchannel);
//     close(ns->fnetd->sockfd);
//     close(ns->fnetd->iwsockfd);
//     g_free(ns->fnetd);
//     g_free(ns->fixcmd);
    g_free(VPNMon_data);
}

GtkWidget *VPNMon_constructor(LXPanel *panel, config_setting_t *settings)
{
    VPNMon_data_s *VPNMon_data;
  
 /* panel is a pointer to the panel and
     settings is a pointer to the configuration data
     since we don't use it, we'll make sure it doesn't
     give us an error at compilation time */
 (void)panel;
 (void)settings;

    // allocate our private structure instance
    VPNMon_data = g_new0(VPNMon_data_s, 1);

    // update the instance count
    VPNMon_data->iMyId = ++iInstanceCount;

 // make a label out of the ID
 char cIdBuf[10] = {'\0'};

 snprintf(cIdBuf, sizeof(cIdBuf), "TP-%d", VPNMon_data->iMyId);

 // create a label widget instance 
 GtkWidget *pLabel = gtk_label_new(cIdBuf);

 // set the label to be visible
 gtk_widget_show(pLabel);

 // need to create a container to be able to set a border
 GtkWidget *p = gtk_event_box_new();

 // our widget doesn't have a window...
 // it is usually illegal to call gtk_widget_set_has_window() from application but for GtkEventBox it doesn't hurt
 gtk_widget_set_has_window(p, FALSE);

 // bind private structure to the widget assuming it should be freed using g_free()
 lxpanel_plugin_set_data(p, VPNMon_data, VPNMon_destructor);

 // set border width
 gtk_container_set_border_width(GTK_CONTAINER(p), 1);

 // add the label to the container
 gtk_container_add(GTK_CONTAINER(p), pLabel);

 // set the size we want
 gtk_widget_set_size_request(p, 40, 25);

    VPNMon_data->pLabel = pLabel;
    
    //******************************//
    VPNMon_data->nls = open_netlink();
    
    //******************************//
    
    VPNMon_data->ttag = g_timeout_add(500, (GSourceFunc)VPNMon_timer1, VPNMon_data);
printf("test_constructor: ttag=%d\n",(int)(VPNMon_data->ttag));

    // success!!!
    return p;
}

#ifdef comment
typedef struct {
    void (*init)(void);         /* optional startup */
    void (*finalize)(void);     /* optional finalize */
    char *name;                 /* name to represent in lists */
    char *description;          /* tooltip text */
    GtkWidget *(*new_instance)(LXPanel *panel, config_setting_t *settings);
    GtkWidget *(*config)(LXPanel *panel, GtkWidget *instance);
    void (*reconfigure)(LXPanel *panel, GtkWidget *instance);
    gboolean (*button_press_event)(GtkWidget *widget, GdkEventButton *event, LXPanel *panel);
    void (*show_system_menu)(GtkWidget *widget);
    gboolean (*update_context_menu)(GtkWidget *plugin, GtkMenu *menu);
    gboolean (*control)(GtkWidget *plugin, const char *cmd); /* not implemented */
    int one_per_system : 1;     /* True to disable more than one instance */
    int expand_available : 1;   /* True if "stretch" option is available */
    int expand_default : 1;     /* True if "stretch" option is default */
    int superseded : 1;         /* True if plugin was superseded by another */
} LXPanelPluginInit;
#endif

/* Plugin descriptor. */
LXPanelPluginInit fm_module_init_lxpanel_gtk = {
   .name = "VPN Monitor",
   .description = "Run a VPN monitor",

   // assigning our functions to provided pointers.
   .new_instance = VPNMon_constructor
};

FM_DEFINE_MODULE(lxpanel_gtk, vpnmon)

